using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Web;
using System.Web.Mvc;
using BMS.Facade;
using BMS.Facade.Data;
using BMS.Facade.Fault;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using BMS.Web.Controllers.Shared;
using BMS.Web.Models;
using DevExpress.Web.Mvc;
using InfoWorld.HL7.ITS;
using Microsoft.Web.Mvc;

namespace BMS.Web.Controllers
{
    /// <summary>
    /// Controller class for the bed board screen.
    /// </summary>
    public class BedBoardController : BaseController
    {
        /// <summary>
        /// Default action for the BedBoardController.
        /// </summary>
        /// <param name="visnIen">The visn ien.</param>
        /// <param name="facilityId">The facility id.</param>
        /// <param name="selectedSort">The selected sort.</param>
        /// <param name="currentPage">The current page.</param>
        /// <returns>
        /// Returns the appropriate view result.
        /// </returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;

            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                BedBoardViewModel model = new BedBoardViewModel();
                model.LoggedUser = this.loggedUser;
                if (string.IsNullOrEmpty(p) || string.IsNullOrEmpty(QueryStrings["visn"]))
                    model.VISN = model.LoggedUser.Visn;
                else
                {
                    model.VISN = FacadeManager.EntityInterface.GetVisn(new II(this.loggedUser.Domain, QueryStrings["visn"]));
                    this.loggedUser.Visn = model.VISN;
                }

                model.VisnExtension = model.VISN.Id.extension;

                bool invalidRequest = false;

                if (!loggedUser.IsSuperUser && loggedUser.SupportUser.IsNo() && loggedUser.NationalUser.IsNo())
                {
                    if (loggedUser.RegionalUser.IsYes())
                        invalidRequest = this.loggedUser.Region.Id.extension != model.VISN.Region.Id.extension;
                    else if (loggedUser.VisnUser.IsYes())
                        invalidRequest = this.loggedUser.Visn.Id.extension != model.VISN.Id.extension;
                }

                if (invalidRequest)
                    return UnauthorizedUrlRequestResult();

                //Fill data            
                model.ReportSelectionList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.SelectReport);
                model.FacilityList = FacadeManager.EntityInterface.GetFacilitiesByVisn(model.VISN.Id).OrderBy(fac => fac.Code).ToList();

                IList<VistaSite> vistaSites = FacadeManager.EntityInterface.GetVistaSites(model.VISN.Id);
                model.BMSBedOccupancyList = new Dictionary<string, BedOccupancyCount>();

                if (vistaSites != null)
                {
                    StringBuilder vistaSitesParameter = new StringBuilder();
                    foreach (VistaSite site in vistaSites)
                    {
                        vistaSitesParameter.Append(site.Id.extension);
                        vistaSitesParameter.Append(",");
                    }

                    model.BMSBedOccupancyList = FacadeManager.BedInterface.GetPercentageOfOccupancy(vistaSitesParameter.ToString(), string.Empty, model.FacilityList.Select(a => a.Id).ToList()).ToDictionary(item => item.FacilityId.extension);
                }

                model.FacilityList = FacadeManager.UserInterface.GetFacilitiesWithReadAccess(this.loggedUser, model.FacilityList.ToList());
                StringBuilder facilities = new StringBuilder();
                foreach (Facility f in model.FacilityList)
                {
                    facilities.Append(f.Id.extension);
                    facilities.Append(",");
                }
                if (facilities.Length > 0)
                    facilities.Remove(facilities.Length - 1, 1);
                model.LoggedUserList = FacadeManager.UserInterface.GetLoggedUsersCount(this.loggedUser.Domain, facilities.ToString());
                model.FacilityList.Insert(0, new Facility() { Id = new II(Strings.AllFacilities, Strings.AllFacilities), Code = Strings.AllFacilities, IsAlive = true });
                model.SelectedFacilityId = string.IsNullOrEmpty(p) ? model.FacilityList[0].Id.extension : string.IsNullOrEmpty(QueryStrings["facility"]) ? model.FacilityList[0].Id.extension : QueryStrings["facility"];
                model.SelectedReportSelectionCode = model.ReportSelectionList[0].code;
                model.PatientTransferList = GetPatientTransferList(model.SelectedFacilityId, model.VISN);
                model.SpecialtyList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.VistaSpecialty).OrderBy(cd => cd.displayName).ToList();

                string firstLoadKey = "IsVISNFirstLoad" + model.VISN.Number;
                if (!TempData.ContainsKey(firstLoadKey))
                    model.IsFirstLoad = true;
                else
                    model.IsFirstLoad = false;

                TempData[firstLoadKey] = false; // Any value will do. If there is any value at this key, we know this site has been loaded during this user session.

                IReportInfo visnDiversionReport = FacadeManager.ReportsInterface.GetReports(FullUserName).Where(reportInfo => reportInfo.Name == "VISN Diversion Report").FirstOrDefault();
                if (visnDiversionReport != null)
                {
                    visnDiversionReport.Parameters = new Dictionary<string, string>();
                    visnDiversionReport.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, this.Request.Url.ToString());
                    visnDiversionReport.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToVISNHomePage);
                    visnDiversionReport.Parameters.Add(Constants.REPORT_TITLE, "VISN Diversion Report");
                    visnDiversionReport.Parameters.Add(Constants.REP_CURRENT_USER, this.loggedUser.UserName);
                    visnDiversionReport.Parameters.Add(Constants.REP_VISN_EXTENSIONS, this.loggedUser.Visn.Id.extension.ToUpper());
                    visnDiversionReport.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION_TO_UPPER, this.loggedUser.Facility.Id.extension.ToUpper());

                    model.DiversionReportLink = visnDiversionReport.Url;
                }
                
                return View(model);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        private BedBoardViewModel BuildModelForGrid()
        {
            BedBoardViewModel model = new BedBoardViewModel();

            model.LoggedUser = this.loggedUser;
            if (!QueryStrings.ContainsKey("visn") || string.IsNullOrEmpty(QueryStrings["visn"]))
                model.VISN = model.LoggedUser.Visn;
            else
                model.VISN = FacadeManager.EntityInterface.GetVisn(new II(this.loggedUser.Domain, QueryStrings["visn"]));
            model.VisnExtension = model.VISN.Id.extension;

            model.FacilityList = FacadeManager.EntityInterface.GetFacilitiesByVisn(model.VISN.Id).OrderBy(fac => fac.Code).ToList();
            model.FacilityList = FacadeManager.UserInterface.GetFacilitiesWithReadAccess(this.loggedUser, model.FacilityList.ToList());
            model.FacilityList.Insert(0, new Facility() { Id = new II(Strings.AllFacilities, Strings.AllFacilities), Code = Strings.AllFacilities, IsAlive = true });
            model.SelectedFacilityId = !QueryStrings.ContainsKey("facility") || string.IsNullOrEmpty(QueryStrings["facility"]) ? model.FacilityList[0].Id.extension : QueryStrings["facility"];

            model.PatientTransferList = GetPatientTransferList(model.SelectedFacilityId, model.VISN);
            model.SpecialtyList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.VistaSpecialty).OrderBy(cd => cd.displayName).ToList();

            return model;
        }

        /// <summary>
        /// Changes the facility.
        /// </summary>
        /// <param name="facilityId">The facility id.</param>
        /// <returns></returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult ChangeFacility(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = FacadeManager.EntityInterface.GetFacility(new II(loggedUser.Domain, QueryStrings["facility"]));
                this.loggedUser.Facility = facility;
                this.loggedUser.VistaSite = facility.VistaSite;
                this.loggedUser.Visn = facility.VistaSite.Visn;
                this.loggedUser.Region = facility.VistaSite.Visn.Region;

                return this.RedirectToAction<HomeController>(act => act.Index());
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Default submit action for the BedBoard screen.
        /// </summary>
        /// <param name="input">The model.</param>
        /// <param name="visnIen">The VISN Ien.</param>
        /// <returns>Returns the appropriate view result.</returns>
        [HttpPost]
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index([Bind(Exclude = "RememberMe")] BedBoardViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (!String.IsNullOrEmpty(input.HeaderCommand))
                {
                    input.LoggedUser = loggedUser;
                    string saveArea = "VISNPPBP" + loggedUser.Visn.Number;
                    string key = loggedUser.Id.extension + saveArea;
                    if (input.HeaderCommand == "Save" && Session[key] != null)
                    {
                        string layoutData = Session[key].ToString();
                        GridHelper.SaveUserLayout(this.loggedUser, saveArea, layoutData, true); // save settings from session to database
                    }
                    else if (input.HeaderCommand == "Reset")
                        GridHelper.SaveUserLayout(this.loggedUser, saveArea, String.Empty, true); // clear saved settings and reset grid
                }

                if (string.IsNullOrEmpty(input.SelectReportOperation))
                    return this.RedirectToAction<BedBoardController>(act => act.Index(EncryptQueryString(new string[] { "visn", "facility" }, new string[] { input.VisnExtension, input.SelectedFacilityId }, loggedUser.Salt)));
                //we need to display a VISN report
                else
                {
                    IList<IReportInfo> rptVisn = FacadeManager.ReportsInterface.GetOtherReports(FullUserName);
                    String reportName = null;
                    IReportInfo report = null;
                    input.LoggedUser = this.loggedUser;
                    string reportTitle = string.Empty;
                    string visnNumber = string.Empty;
                    visnNumber = input.LoggedUser == null ? string.Empty : input.LoggedUser.Visn == null ? string.Empty : input.LoggedUser.Visn.Number;

                    //locate the VISN report
                    switch (input.SelectedReportSelectionCode)
                    {
                        case Constants.SELECT_REPORT_ACTIVE:
                            reportName = Constants.VISN_ACTIVE_RPT;
                            report = rptVisn.Where(a => a.Name == reportName).FirstOrDefault();
                            reportTitle = string.Format(Strings.VISNReportTitleActive, visnNumber);
                            break;
                        case Constants.SELECT_REPORT_DISP:
                            reportName = Constants.VISN_DISPOSITION_RPT;
                            report = rptVisn.Where(a => a.Name == reportName).FirstOrDefault();
                            reportTitle = string.Format(Strings.VISNReportTitleDisposition, visnNumber);
                            break;
                        case Constants.SELECT_REPORT_CONTRACT:
                            reportName = Constants.VISN_CONTRACT_RPT;
                            report = rptVisn.Where(a => a.Name == reportName).FirstOrDefault();
                            reportTitle = string.Format(Strings.VISNReportTitleContract, visnNumber);
                            break;
                        case Constants.SELECT_REPORT_DATE_AUDIT:
                            reportName = Constants.VISN_AUDIT_RPT;
                            report = rptVisn.Where(a => a.Name == reportName).FirstOrDefault();
                            reportTitle = string.Format(Strings.VISNReportTitleAudit, visnNumber);
                            break;
                    }
                    //Redirect to the report
                    if (report == null)
                        throw new ReportRenderException(reportName, Strings.ERR_ReportNotAvailable);
                    else
                    {
                        if (report.Parameters == null)
                            report.Parameters = new Dictionary<String, String>();
                        report.Parameters.Add(Constants.REP_TIME_ZONE_MINUTES_OFFSET, RepTimeZoneMinutesOffset);
                        report.Parameters.Add(Constants.REP_VISN_NETWORK_VISN_EXTENSION, string.IsNullOrEmpty(input.VisnExtension) ? input.LoggedUser.Visn.Id.extension.ToString() : input.VisnExtension);
                        report.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, this.Request.Url.ToString());
                        report.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToBedBoardHomePage);
                        report.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                        return Redirect(report.Url);
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        [HttpPost]
        public ActionResult CustomAction(List<string> actIds, string action)
        {
            switch (action)
            {
                case "set_national":
                    {
                        if (!FacadeManager.UserInterface.CheckPermission("Transfer, EditTransfer Update"))
                            return BaseController.UnauthorizedWriteRequestResult(loggedUser.Facility.Name);

                        foreach (string actId in actIds)
                        {
                            try
                            {
                                II transferId = new II(loggedUser.Domain, actId);
                                Transfer transfer = FacadeManager.ADTInterface.GetTransferById(transferId, loggedUser);
                                if (transfer == null)
                                    continue;

                                transfer.IsNationalWaitList = !transfer.IsNationalWaitList;
                                FacadeManager.WorkflowInterface.UpdateTransferRequest(transfer, transfer.FacilityFrom);
                            }
                            catch
                            {
                                // errors will be logged at the service layer if GetTransferById or UpdateTransferRequest have issues. Meanwhile other transfers in the loop should be processed
                            }
                        }
                        break;
                    }
                default:
                    break;
            }

            return BedBoardGrid(null);
        }

        public ActionResult ExportTo(string p)
        {
            string outputFormat = QueryStrings["outputFormat"];
            BedBoardViewModel model = BuildModelForGrid();

            GridViewSettings settings = GridHelper.CreateVISNPPBPGrid(null, model, true);
            switch (outputFormat.ToUpper())
            {
                case "CSV":
                    return GridViewExtension.ExportToCsv(settings, model.PatientTransferList);
                case "PDF":
                    return GridViewExtension.ExportToPdf(settings, model.PatientTransferList);
                case "RTF":
                    return GridViewExtension.ExportToRtf(settings, model.PatientTransferList);
                case "XLS":
                    return GridViewExtension.ExportToXls(settings, model.PatientTransferList);
                case "XLSX":
                    return GridViewExtension.ExportToXlsx(settings, model.PatientTransferList);
                default:
                    return RedirectToAction("BedBoardGrid");
            }
        }

        /// <summary>
        /// Shows the VISN Bed Summary Report.
        /// </summary>
        /// <param name="visnRoot">The visn root.</param>
        /// <param name="visnExtension">The visn extension.</param>
        /// <param name="visnNumber">The visn number.</param>
        /// <returns>Displays VISN Bed Summary Report.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult ShowVISNBedSummaryReport(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                II visnId = new II { root = loggedUser.Domain, extension = QueryStrings["visnExtension"] };
                IList<Facility> facilites = FacadeManager.EntityInterface.GetFacilitiesByVisn(visnId).Where(a => a.IsAlive == true).ToList();
                string parameters = string.Empty;
                if (facilites != null)                
                    parameters = FacadeManager.ConfigurationInterface.GetFacMedicalCenterIds(this.loggedUser.Domain, facilites.Select(a => a.Id).ToList());                

                IReportInfo visnSummaryReport = FacadeManager.ReportsInterface.GetReports(FullUserName).Where(a => a.Name == Constants.VISNBedSummaryReport).FirstOrDefault();
                if (visnSummaryReport != null)
                {
                    //create paramets fro report
                    if (visnSummaryReport.Parameters == null)
                        visnSummaryReport.Parameters = new Dictionary<String, String>();

                    visnSummaryReport.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToBedBoardHomePage);
                    visnSummaryReport.Parameters.Add(Constants.REPORT_TITLE, string.Format("{0} {1} {2}", Strings.VISN.ToUpper(), QueryStrings["visnNumber"], Strings.BedSummaryReportTitle));
                    visnSummaryReport.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, QueryStrings["returnUrl"]);
                    visnSummaryReport.Parameters.Add(Constants.REP_GENERAL_FACILITIES, parameters);
                    visnSummaryReport.Parameters.Add(Constants.REP_VISN_BED_SUMMARY_VISN_ROOT, loggedUser.Domain);
                    visnSummaryReport.Parameters.Add(Constants.REP_VISN_BED_SUMMARY_VISN_EXTENSION, QueryStrings["visnExtension"]);
                    visnSummaryReport.Parameters.Add(Constants.REP_USER_SALT, loggedUser.Salt);
                    
                    //Redirect to the report
                    return Redirect(visnSummaryReport.Url);
                }
                else
                    throw new ReportRenderException(string.Format("{0} {1} {2}", Strings.VISN.ToUpper(), QueryStrings["visnNumber"], Strings.BedSummaryReportTitle), Strings.ERR_ReportNotAvailable);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Shows the facility bed summary report.
        /// </summary>
        /// <param name="facilityRoot">The facility root.</param>
        /// <param name="facilityExtension">The facility extension.</param>
        /// <param name="facilityName">Name of the facility.</param>
        /// <param name="reportName">Name of the report.</param>
        /// <returns>Displays facility bed summary report.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult ShowFacilityBedSummaryReport(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IReportInfo bedSummaryReport = Facade.FacadeManager.ReportsInterface.GetReports(FullUserName).Where(a => a.Name == Constants.BED_SUMMARY_REPORT).FirstOrDefault();
                if (bedSummaryReport != null)
                {
                    if (bedSummaryReport.Parameters == null)
                        bedSummaryReport.Parameters = new Dictionary<String, String>();
                    bedSummaryReport.Parameters.Add(Constants.REP_GENERAL_RETURN_PATH, QueryStrings["returnUrl"]);
                    bedSummaryReport.Parameters.Add(Constants.REP_GENERAL_RETURN_TEXT, Strings.ReturnToBedBoardHomePage);

                    DateTime dtUtc = DateTime.UtcNow;
                    TimeZoneInfo tzFac = FacadeUtil.GetFacilityTimeZoneInfoByFacilityId(new II(loggedUser.Domain, QueryStrings["facilityExtension"]));
                    DateTime dtTz = TimeZoneInfo.ConvertTimeFromUtc(dtUtc, tzFac);
                    string offset = dtTz.Subtract(dtUtc).TotalMinutes.ToString();
                    
                    bedSummaryReport.Parameters.Add(Constants.REP_TIME_ZONE_MINUTES_OFFSET, offset);

                    FacilitySettings settings = FacadeManager.ConfigurationInterface.GetFacilitySettings(this.loggedUser.Facility.Id);

                    if (!string.IsNullOrEmpty(settings.MedicalCenterID) && settings.IntegratedFacility != null &&
                        settings.IntegratedFacility.code.ToUpper() == "YES")
                        bedSummaryReport.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, settings.MedicalCenterID);
                    else
                        bedSummaryReport.Parameters.Add(Constants.REP_SCHEDULED_ADMISSION_MEDICAL_DIVISION, "");

                    bedSummaryReport.Parameters.Add(Constants.REP_GENERAL_FACILITY_EXTENSION, QueryStrings["facilityExtension"]);
                    string reportTitle = string.Empty;
                    reportTitle = Strings.BedSummaryReportTitle + ": " + QueryStrings["facilityName"];
                    bedSummaryReport.Parameters.Add(Constants.REPORT_TITLE, reportTitle);
                    bedSummaryReport.Parameters.Add(Constants.REP_USER_SALT, loggedUser.Salt);
                    //Redirect to the report
                    return Redirect(bedSummaryReport.Url);
                }
                else
                    throw new ReportRenderException(QueryStrings["reportName"], Strings.ERR_ReportNotAvailable);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #region Private Methods

        /// <summary>
        /// Gets the patient transfer list.
        /// </summary>
        /// <param name="facilityIdExtension">The facility id extension.</param>        
        /// <param name="sortBy">The sort by.</param>
        /// <param name="visnId">The VISN id.</param>
        /// <param name="pageSize">The page size.</param>
        /// <param name="pageNumber">The page number.</param>
        /// <returns>Patient transfer list.</returns>
        private IList<TransferInfo> GetPatientTransferList(string facilityIdExtension, Visn visn)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Facility facility = null;
                if (facilityIdExtension != Strings.AllFacilities)
                    facility = FacadeManager.EntityInterface.GetFacility(new II(this.loggedUser.Domain, facilityIdExtension));                

                return FacadeManager.ADTInterface.GetPatientsInCommunityHospitals(facility, visn);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        #endregion

        [HttpPost]
        [ValidateInput(false)]
        public ActionResult BedBoardGrid([Bind(Exclude = "RememberMe")] BedBoardViewModel model)
        {
            if (model == null || model.PatientTransferList == null)
                model = BuildModelForGrid();

            return PartialView("BedBoardGrid", model);
        }
    }
}
